//
// Bitmovin Player iOS SDK
// Copyright (C) 2021, Bitmovin GmbH, All Rights Reserved
//
// This source code and its use and distribution, is subject to the terms
// and conditions of the applicable license agreement.
//

#import <Foundation/Foundation.h>

NS_ASSUME_NONNULL_BEGIN

FOUNDATION_EXPORT NSErrorDomain const BMPBitmovinSourceErrorDomain NS_SWIFT_NAME(BitmovinSourceErrorDomain);

/**
 * This enum represents the error codes that can be associated with ``BitmovinPlayerCore/SourceErrorEvent``.
 */
typedef NS_ERROR_ENUM(BMPBitmovinSourceErrorDomain, BMPSourceErrorCode) {
    /**
     * A general source error code when a more specific error code could not be determined.
     *
     * See ``BitmovinPlayerCore/SourceErrorEvent/message`` for the error message.
     */
    BMPSourceErrorCodeGeneral = 2001,
    /**
     * Failed to start playing the source.
     *
     * See ``BitmovinPlayerCore/SourceErrorEvent/message`` for the error message.
     */
    BMPSourceErrorCodePlaybackFailed = 2101,
    /**
     * Downloading the playlist has failed, but no additional details are available.
     *
     * To configure retry handling for network requests, see ``BitmovinPlayerCore/NetworkConfig/retryHttpRequest``.
     *
     * See our guide on the Network API for more information [here](https://developer.bitmovin.com/playback/docs/ios-network-api).
     *
     * See ``BitmovinPlayerCore/SourceErrorEvent/message`` for the error message.
     */
    BMPSourceErrorCodeManifestDownloadFailed = 2202,
    /**
     * Downloading the playlist has failed due to an invalid manifest URL.
     *
     * The manifest URL is either malformed or empty.
     * Try to load another manifest URL.
     *
     * See ``BitmovinPlayerCore/SourceErrorEvent/message`` for the error message.
     */
    BMPSourceErrorCodeManifestUrlInvalid = 2205,
    /**
     * Downloading the playlist has failed due to an unhandled HTTP redirect (3xx).
     *
     * The manifest request returned a 3xx HTTP status code which could not be handled.
     *
     * Make sure that your server is returning valid HTTP status codes for redirects.
     *
     * See ``BitmovinPlayerCore/SourceErrorEvent/message`` for the error message.
     */
    BMPSourceErrorCodeManifestUnhandledRedirect = 2210,
    /**
     * Downloading the playlist has failed due to a client-side HTTP error (4xx).
     *
     * The manifest request returned a 4xx HTTP status code which is not explicitly handled with a specific
     * error code.
     *
     * Make sure to use the correct URL and that the server is reachable.
     * 
     * See our guide on the Network API for more information [here](https://developer.bitmovin.com/playback/docs/ios-network-api).
     *
     * See ``BitmovinPlayerCore/SourceErrorEvent/message`` for the error message.
     */
    BMPSourceErrorCodeManifestClientError = 2220,
    /**
     * Downloading the playlist has failed due to a bad request.
     * Received HTTP status code 400 - Bad Request.
     *
     * Make sure to use the correct URL and that the server is reachable.
     *
     * To configure pre-processing of network requests, see ``BitmovinPlayerCore/NetworkConfig/preprocessHttpRequest``.
     *
     * See our guide on the Network API for more information [here](https://developer.bitmovin.com/playback/docs/ios-network-api).
     *
     * See ``BitmovinPlayerCore/SourceErrorEvent/message`` for the error message.
     */
    BMPSourceErrorCodeManifestBadRequest = 2221,
    /**
     * Downloading the playlist has failed due to missing or invalid authentication credentials.
     * Received HTTP status code 401 - Unauthorized.
     *
     * The manifest URL might require authentication.
     *
     * To add headers to the manifest request, see ``BitmovinPlayerCore/NetworkConfig/preprocessHttpRequest``.
     * 
     * See our guide on the Network API for more information [here](https://developer.bitmovin.com/playback/docs/ios-network-api).
     *
     * See ``BitmovinPlayerCore/SourceErrorEvent/message`` for the error message.
     */
    BMPSourceErrorCodeManifestNotAuthorized = 2222,
    /**
     * Downloading the playlist has failed due to insufficient permissions.
     * Received HTTP status code 403 - Forbidden.
     *
     * The manifest URL might require authentication.
     *
     * To add headers to the manifest request, see ``BitmovinPlayerCore/NetworkConfig/preprocessHttpRequest``.
     * 
     * See our guide on the Network API for more information [here](https://developer.bitmovin.com/playback/docs/ios-network-api).
     *
     * See ``BitmovinPlayerCore/SourceErrorEvent/message`` for the error message.
     */
    BMPSourceErrorCodeManifestNotAllowed = 2223,
    /**
     * Downloading the playlist has failed because the requested manifest could not be found.
     * Received HTTP status code 404 - Not Found.
     *
     * Make sure to use the correct URL and that the server is reachable.
     *
     * To configure pre-processing of network requests, see ``BitmovinPlayerCore/NetworkConfig/preprocessHttpRequest``.
     *
     * See our guide on the Network API for more information [here](https://developer.bitmovin.com/playback/docs/ios-network-api).
     *
     * See ``BitmovinPlayerCore/SourceErrorEvent/message`` for the error message.
     */
    BMPSourceErrorCodeManifestNotFound = 2224,
    /**
     * Downloading the playlist has failed due to a general server-side issue (5xx).
     *
     * The manifest request returned a 5xx HTTP status code which is not explicitly handled with a specific
     * error code.
     *
     * Make sure to use the correct URL and that the server is reachable.
     *
     * To configure pre-processing of network requests, see ``BitmovinPlayerCore/NetworkConfig/preprocessHttpRequest``.
     *
     * To configure retry handling for network requests, see ``BitmovinPlayerCore/NetworkConfig/retryHttpRequest``.
     * 
     * See our guide on the Network API for more information [here](https://developer.bitmovin.com/playback/docs/ios-network-api).
     *
     * See ``BitmovinPlayerCore/SourceErrorEvent/message`` for the error message.
     */
    BMPSourceErrorCodeManifestGeneralServerError = 2230,
    /**
     * Downloading the playlist has failed due to an internal server error.
     * Received HTTP status code 500 - Internal Server Error.
     *
     * Make sure to use the correct URL and that the server is reachable.
     *
     * To configure retry handling for network requests, see ``BitmovinPlayerCore/NetworkConfig/retryHttpRequest``.
     * 
     * See our guide on the Network API for more information [here](https://developer.bitmovin.com/playback/docs/ios-network-api).
     *
     * See ``BitmovinPlayerCore/SourceErrorEvent/message`` for the error message.
     */
    BMPSourceErrorCodeManifestServerError = 2231,
    /**
     * Downloading the playlist has failed due to a bad gateway error.
     * Received HTTP status code 502 - Bad Gateway.
     *
     * Make sure to use the correct URL and that the server is reachable.
     *
     * To configure retry handling for network requests, see ``BitmovinPlayerCore/NetworkConfig/retryHttpRequest``.
     * 
     * See our guide on the Network API for more information [here](https://developer.bitmovin.com/playback/docs/ios-network-api).
     *
     * See ``BitmovinPlayerCore/SourceErrorEvent/message`` for the error message.
     */
    BMPSourceErrorCodeManifestServerGatewayError = 2232,
    /**
     * Downloading the playlist has failed because the server is temporarily unavailable.
     * Received HTTP status code 503 - Service Unavailable.
     *
     * Make sure to use the correct URL and that the server is reachable.
     *
     * To configure retry handling for network requests, see ``BitmovinPlayerCore/NetworkConfig/retryHttpRequest``.
     * 
     * See our guide on the Network API for more information [here](https://developer.bitmovin.com/playback/docs/ios-network-api).
     *
     * See ``BitmovinPlayerCore/SourceErrorEvent/message`` for the error message.
     */
    BMPSourceErrorCodeManifestServerUnavailable = 2233,
    /**
     * Downloading the playlist has failed due to a gateway timeout.
     * Received HTTP status code 504 - Gateway Timeout.
     *
     * Make sure to use the correct URL and that the server is reachable.
     *
     * To configure retry handling for network requests, see ``BitmovinPlayerCore/NetworkConfig/retryHttpRequest``.
     * 
     * See our guide on the Network API for more information [here](https://developer.bitmovin.com/playback/docs/ios-network-api).
     *
     * See ``BitmovinPlayerCore/SourceErrorEvent/message`` for the error message.
     */
    BMPSourceErrorCodeManifestServerGatewayTimeout = 2234,
    /**
     * A general error indicating that the current DRM session failed for some reason.
     *
     * Make sure to configure ``BitmovinPlayerCore/SourceConfig/drmConfig`` correctly.
     * 
     * See our guide on DRM for more information [here](https://developer.bitmovin.com/playback/docs/drm-providers-setup-ios).
     *
     * See ``BitmovinPlayerCore/SourceErrorEvent/message`` for the error message.
     */
    BMPSourceErrorCodeDrmGeneral = 2301,
    /**
     * The required DRM configuration is missing for protected content.
     *
     * Make sure to configure ``BitmovinPlayerCore/SourceConfig/drmConfig`` correctly.
     * 
     * See our guide on DRM for more information [here](https://developer.bitmovin.com/playback/docs/drm-providers-setup-ios).
     */
    BMPSourceErrorCodeDrmConfigMissing = 2302,
    /**
     * No DRM license URL provided.
     *
     * The player was unable to determine the DRM license URL from either ``BitmovinPlayerCore/FairplayConfig``
     * or from the HLS manifest's [`EXT-X-KEY`](https://datatracker.ietf.org/doc/html/draft-pantos-hls-rfc8216bis) tag's `URI` attribute.
     * 
     * See our guide on DRM for more information [here](https://developer.bitmovin.com/playback/docs/drm-providers-setup-ios).
     */
    BMPSourceErrorCodeNoDrmLicenseServerUrlProvided = 2303,
    /**
     * No DRM certificate URL provided.
     *
     * The player was unable to determine the DRM license URL from ``BitmovinPlayerCore/FairplayConfig/certificateUrl``.
     * 
     * See our guide on DRM for more information [here](https://developer.bitmovin.com/playback/docs/drm-providers-setup-ios).
     */
    BMPSourceErrorCodeNoDrmCertificateServerUrlProvided = 2309,
    /**
     * The response to the DRM license request failed with an invalid HTTP response code.
     *
     * Make sure to use the correct URL and that the server is reachable.
     * For adding custom headers to the DRM license request, see ``BitmovinPlayerCore/FairplayConfig/licenseRequestHeaders``.
     * 
     * See our guide on DRM for more information [here](https://developer.bitmovin.com/playback/docs/drm-providers-setup-ios).
     *
     * See ``BitmovinPlayerCore/SourceErrorEvent/message`` for the error message.
     */
    BMPSourceErrorCodeDrmLicenseRequestFailed = 2306,
    /**
     * The response to the DRM certificate request failed with an invalid HTTP response code.
     *
     * Make sure to use the correct URL and that the server is reachable.
     * For adding custom headers to the DRM certificate request, see ``BitmovinPlayerCore/FairplayConfig/certificateRequestHeaders``.
     * 
     * See our guide on DRM for more information [here](https://developer.bitmovin.com/playback/docs/drm-providers-setup-ios).
     *
     * See ``BitmovinPlayerCore/SourceErrorEvent/message`` for the error message.
     */
    BMPSourceErrorCodeDrmCertificateRequestFailed = 2307
} NS_SWIFT_NAME(SourceError);

NS_ASSUME_NONNULL_END
